<?php

namespace App\Http\Controllers;

use App\Models\Organization;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Http\Requests\OrganizationRequest;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class OrganizationController extends Controller
{
    use AuthorizesRequests;

    public function index()
    {
        return Inertia::render('Organizations/Index', [
            'organizations' => auth()->user()->organizations()
                ->withCount('users')
                ->latest()
                ->paginate(10),
        ]);
    }

    public function create()
    {
        return Inertia::render('Organizations/Create');
    }

    public function store(OrganizationRequest $request)
    {
        try {
            // Check if name exists (excluding soft deleted)
            $nameExists = Organization::where('name', $request->name)
                ->whereNull('deleted_at')
                ->exists();

            if ($nameExists) {
                return back()->withInput()
                    ->withErrors(['name' => 'An organization with this name already exists.']);
            }

            $user = auth()->user();
            if($user->phone_number == null){
                // Update user's phone number
                $user->update([
                    'phone_number' => $request->phone_number,
                ]);
            }

            $organization = Organization::create([
                'name' => $request->name,
                'description' => $request->description,
                'owner_id' => $user->id,
                'emergency_contact_id' => $user->id, // Set creator as emergency contact
                'organization_type' => $request->organization_type,
                'industries' => $request->industries,
                'employee_count_range' => $request->employee_count_range,
                'region' => $request->region,
                'timezone' => $request->timezone,
                'primary_aws_region' => $request->primary_aws_region,
                'compliance_requirements' => $request->compliance_requirements,
                'aws_org_role' => $request->aws_org_role,
                'tags' => $request->tags,
                'onboarding_goal' => $request->onboarding_goal,
            ]);

            $organization->users()->attach($user->id, [
                'role' => 'owner',
                'joined_at' => now(),
            ]);

            return redirect()->route('organizations.show', $organization)
                ->with('success', 'Organization created successfully.');
        } catch (\Exception $e) {
            return back()->withInput()
                ->with('error', 'Unable to create organization. Please try again.');
        }
    }

    public function show(Organization $organization)
    {
        try {
            $this->authorize('view', $organization);

            return Inertia::render('Organizations/Show', [
                'organization' => $organization->load([
                    'owner', 
                    'emergencyContact',
                    'users' => fn($query) => $query->select('users.id', 'name', 'email', 'phone_number')
                ]),
                'userRole' => $organization->users()->where('user_id', auth()->id())->value('role'),
                'canManageMembers' => auth()->user()->can('manageMembers', $organization),
            ]);
        } catch (\Exception $e) {
            return redirect()->route('organizations.index')
                ->with('error', 'You do not have permission to view this organization.');
        }
    }

    public function edit(Organization $organization)
    {
        $this->authorize('update', $organization);

        return Inertia::render('Organizations/Edit', [
            'organization' => $organization->load([
                'owner',
                'emergencyContact',
                'users' => fn($query) => $query->select('users.id', 'name', 'email', 'phone_number')
            ]),
        ]);
    }

    public function update(OrganizationRequest $request, Organization $organization)
    {
        try {
            $this->authorize('update', $organization);
            
            // Check if name exists (excluding soft deleted)
            $nameExists = Organization::where('name', $request->name)
                ->where('id', '!=', $organization->id)
                ->whereNull('deleted_at')
                ->exists();

            if ($nameExists) {
                return back()->withInput()
                    ->withErrors(['name' => 'An organization with this name already exists.']);
            }

            $organization->update($request->validated());
            $user = auth()->user();
            if($user->phone_number == null){
                $user->update([
                    'phone_number' => $request->phone_number,
                ]);
            }
            return redirect()->route('organizations.show', $organization)
                ->with('success', 'Organization updated successfully.');
        } catch (\Exception $e) {
            return back()->withInput()
                ->withErrors(['name' => 'Unable to update organization. Please try again.']);
        }
    }

    public function destroy(Organization $organization)
    {
        $this->authorize('delete', $organization);

        $organization->delete();

        return redirect()->route('organizations.index')
            ->with('success', 'Organization deleted successfully.');
    }

    public function removeMember(Organization $organization, User $user)
    {
        $this->authorize('manageMembers', $organization);

        if ($user->id === $organization->owner_id) {
            return back()->with('error', 'Cannot remove the organization owner.');
        }

        $organization->users()->detach($user->id);

        return back()->with('success', 'Member removed successfully.');
    }

    public function updateMemberRole(Organization $organization, User $user, Request $request)
    {
        $this->authorize('manageMembers', $organization);

        $validated = $request->validate([
            'role' => ['required', 'string', 'in:admin,editor,readonly'],
        ]);

        if ($user->id === $organization->owner_id) {
            return back()->with('error', 'Cannot change the owner\'s role.');
        }

        $organization->users()->updateExistingPivot($user->id, [
            'role' => $validated['role'],
        ]);

        return back()->with('success', 'Member role updated successfully.');
    }

    public function constants()
    {
        return response()->json([
            'ORGANIZATION_TYPES' => Organization::ORGANIZATION_TYPES,
            'EMPLOYEE_RANGES' => Organization::EMPLOYEE_RANGES,
            'AWS_ORG_ROLES' => Organization::AWS_ORG_ROLES,
            'INDUSTRIES' => Organization::INDUSTRIES,
            'ONBOARDING_GOALS' => Organization::ONBOARDING_GOALS,
        ]);
    }
} 