<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Organization extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'owner_id',
        'organization_type',
        'industries',
        'employee_count_range',
        'region',
        'timezone',
        'primary_aws_region',
        'compliance_requirements',
        'aws_org_role',
        'tags',
        'onboarding_goal',
        'emergency_contact_id',
    ];

    protected $casts = [
        'compliance_requirements' => 'array',
        'tags' => 'array',
        'industries' => 'array',
    ];

    // Add constants for dropdown options
    public const ORGANIZATION_TYPES = [
        'enterprise' => 'Enterprise',
        'startup' => 'Startup',
        'educational' => 'Educational Institution',
        'government' => 'Government',
        'non_profit' => 'Non-Profit',
    ];

    public const EMPLOYEE_RANGES = [
        '1-10' => '1-10 employees',
        '11-50' => '11-50 employees',
        '51-200' => '51-200 employees',
        '201-500' => '201-500 employees',
        '501-1000' => '501-1000 employees',
        '1001-5000' => '1001-5000 employees',
        '5000+' => '5000+ employees',
    ];

    public const AWS_ORG_ROLES = [
        'admin' => 'Administrator',
        'member' => 'Member',
        'none' => 'Not Part of AWS Organization',
    ];

    public const INDUSTRIES = [
        'technology' => 'Technology',
        'healthcare' => 'Healthcare',
        'finance' => 'Finance',
        'education' => 'Education',
        'retail' => 'Retail',
        'manufacturing' => 'Manufacturing',
        'government' => 'Government',
        'non_profit' => 'Non-Profit',
        'other' => 'Other',
    ];

    public const ONBOARDING_GOALS = [
        'security_monitoring' => 'Security Monitoring',
        'compliance_audit' => 'Compliance Audit',
        'cost_optimization' => 'Cost Optimization',
        'performance_monitoring' => 'Performance Monitoring',
        'incident_response' => 'Incident Response',
        'other' => 'Other',
    ];

    public const COMPLIANCE_STANDARDS = [
        'hipaa' => [
            'name' => 'HIPAA',
            'description' => 'Health Insurance Portability and Accountability Act',
            'category' => 'Healthcare',
        ],
        'gdpr' => [
            'name' => 'GDPR',
            'description' => 'General Data Protection Regulation',
            'category' => 'Data Protection',
        ],
        'pci_dss' => [
            'name' => 'PCI DSS',
            'description' => 'Payment Card Industry Data Security Standard',
            'category' => 'Financial',
        ],
        'sox' => [
            'name' => 'SOX',
            'description' => 'Sarbanes-Oxley Act',
            'category' => 'Financial',
        ],
        'iso_27001' => [
            'name' => 'ISO 27001',
            'description' => 'Information Security Management System Standard',
            'category' => 'Security',
        ],
        'nist' => [
            'name' => 'NIST',
            'description' => 'National Institute of Standards and Technology',
            'category' => 'Security',
        ],
        'fedramp' => [
            'name' => 'FedRAMP',
            'description' => 'Federal Risk and Authorization Management Program',
            'category' => 'Government',
        ],
        'ccpa' => [
            'name' => 'CCPA',
            'description' => 'California Consumer Privacy Act',
            'category' => 'Data Protection',
        ],
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($organization) {
            $organization->slug = static::createUniqueSlug($organization->name);
        });

        static::updating(function ($organization) {
            if ($organization->isDirty('name')) {
                $organization->slug = static::createUniqueSlug($organization->name, $organization->id);
            }
        });
    }

    protected static function createUniqueSlug($name, $ignoreId = null)
    {
        $slug = Str::slug($name);
        $count = 0;

        // Generate a unique slug
        while (static::checkSlugExists($slug, $count, $ignoreId)) {
            $count++;
            $slug = Str::slug($name) . '-' . $count;
        }

        return $count ? $slug : Str::slug($name);
    }

    protected static function checkSlugExists($slug, $count, $ignoreId = null)
    {
        $query = static::where('slug', $count ? $slug : Str::slug($slug))
            ->whereNull('deleted_at');

        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }

    public function owner()
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    public function users()
    {
        return $this->belongsToMany(User::class)
            ->withPivot(['role', 'invited_at', 'joined_at'])
            ->withTimestamps();
    }

    public function isOwner(User $user): bool
    {
        return $this->owner_id === $user->id;
    }

    public function isMember(User $user): bool
    {
        return $this->users()->where('user_id', $user->id)->exists();
    }

    public function scopeWithMembershipStatus($query, $userId)
    {
        return $query->with(['users' => function ($query) use ($userId) {
            $query->where('users.id', $userId);
        }]);
    }

    public function emergencyContact()
    {
        return $this->belongsTo(User::class, 'emergency_contact_id');
    }

    public function invitations()
    {
        return $this->hasMany(OrganizationInvitation::class);
    }

    protected $appends = ['pending_invitation_count'];

    public function getPendingInvitationCountAttribute()
    {
        return $this->invitations()
            ->whereNull('accepted_at')
            ->whereNull('declined_at')
            ->where('expires_at', '>', now())
            ->count();
    }
} 